/* 
 * Fast CPU cache optimized prime number generator.
 *
 * Calculates primes upto 2^64.
 * Primes can be computed for a given range of numbers.
 * Sieve (page) size can be specified to correspond to the CPU cache size for better performance.
 *
 * (c) 2008  Girish Rao
 */

#include <errno.h>
#include <getopt.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "primes.h"

char	*name;

void usage(void);

int
main(int argc, char *argv[])
{
	Uns32	sbytes;	
	Uns64	hi, lo;
	extern char	*optarg;
	extern int	optind, optopt;
	int	c, ret, tflag, fflag;
	FILE	*fp;

	fp = stdout;
	hi = lo = 0;
	sbytes = 0;
	tflag = fflag = 0;

	if ((name = strrchr(argv[0], '/')) != NULL)
		name++;	

	while ((c = getopt(argc, argv, "hs:f:t")) != -1) {
		switch (c) {
		case 'f':
			if (tflag)
				usage();
			fflag = 1;
			if ((fp = fopen(optarg, "w")) == NULL) {
				perror("Error: fopen()");
				exit(1);
			}
			break;
		case 't':
			if (fflag)
				usage();
			tflag = 1;
			fp = NULL;
			break;
		case 's':
			errno = 0;
			sbytes = (int)strtol(optarg, NULL, 10) * 1024;
			if (!sbytes || errno == ERANGE || sbytes > UINT_MAX>>4) {
				fprintf(stderr, "Error: Sieve size is too" \
				    " large or invalid\n");
				exit(errno ? errno : 1);
			}
			break;
		case 'h':
			usage();
			break;
		case '?':
			fprintf(stderr, "Unrecognised option: -%c\n\n", optopt);
			usage();
			break;
		}
	}
	switch (argc - optind) {
	case 2:
		errno = 0;
#ifdef LLONG
		lo = strtoull(argv[optind], NULL, 10);
#else
		lo = strtoul(argv[optind], NULL, 10);
#endif
		if (lo == 0 || errno == ERANGE) {
			fprintf(stderr, "Error: Number is out of range" \
			    " or invalid\n");
			exit(1);
		}
		optind++;
	case 1:
		errno = 0;
#ifdef LLONG
		hi = strtoull(argv[optind], NULL, 10);
#else
		hi = strtoul(argv[optind], NULL, 10);
#endif
		if (hi < 1 || errno == ERANGE) {
			fprintf(stderr, "Error: Number is out of range" \
			    " or invalid\n");
			exit(1);
		}
		break;
	default:	
		usage();
	}

	ret = primes(hi, lo, sbytes, fp);
	if (ret < 0)
		exit(1);

	return 0;
}

void
usage(void)
{

	fprintf(stderr, 
	    "Usage: %s [-t | -f file] [-s kb] [low] high\n\n"
	    "\t-t test\n"
	    "\t-f save primes to file\n"
	    "\t-s sieve size in kilobytes\n"
		,name);

	exit(2);
}
