/* 
 * (c) 2008  Girish Rao
 */

#include <sys/time.h>

#include <errno.h>
#include <getopt.h>
#include <limits.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "primes.h"

/* check if the number n is marked as prime */
#define TEST(p,n)       (*(p + (n >> 4)) & (1 << ((n >> 1) & 7)))

/* set n as composite */
#define SET(p,n)        *(p + (n >> 4)) |= 1 << ((n >> 1) & 7)

static Uns8	*sieve;		 
static Uns32	*svprimes;	/* stores sieving primes */

static void pre_sieve(Uns32 , int , Uns32 , Uns64* , Uns32* , Uns64* ,FILE*);
static void do_sieve(Uns32 , int , Uns64, Uns64, Uns32, Uns64* ,FILE*);
static inline void print(Uns32 , FILE* ); 

Int64
primes(Uns64 high, Uns64 low, Uns32 sbytes, FILE *fp)
{
	Uns32	ssize;		/* numbers in the sieve */
	Uns32	maxsvpr;	/* largest sieving prime */
	Uns32	nprimes;	/* number of sieving primes */
	Uns64	totpr;		/* primes found */
	Uns32	pbytes;		/* size of svprimes[] */
	int	prnt;

	struct timeval	tps, tpe; 

	if (low > high || high < 2) 
		return 0;

	maxsvpr = sqrt((double)high);   /* size of max sieving prime */
	nprimes = totpr = 0;
	fp == NULL ? (prnt = 0) : (prnt = 1);
	if (sbytes) {
		ssize = sbytes * 16;
		if (ssize < maxsvpr) {
			fprintf(stderr, "Error: sieve size too small. " \
			   "min required %dKB\n", (int)ceil(maxsvpr / 16 / 1024) + 1);
			return -1;
		}
	} else {
		sbytes = 16 << 10;	/* default sieve size 16 kb */
		ssize = sbytes * 16;
		if (ssize < maxsvpr) {
			ssize = maxsvpr;
			sbytes = ssize >> 4;
		}
	}
	if (high < ssize)	/* we don't want a sieve larger than hi */
        	ssize = high;
	if (!(low & 1))
		++low;

	/* 
	 * calculate pi function 
	 */
	if (maxsvpr > 10) {
        	pbytes = floor((1.015 * (maxsvpr)) / (log((double)(maxsvpr)) - 1)) \
		    * sizeof(int);
	} else { 
		pbytes = 4 * sizeof(int);
	}

	if ((sieve = malloc(sbytes)) == NULL) {
		perror("Error: malloc(sieve)");
		exit(1);
	}
	if ((svprimes = malloc(pbytes)) == NULL) {
		perror("Error: malloc(svprimes)");
		exit(1);
	}
	if (fp != stdout) {
        	if (gettimeofday(&tps, NULL) < 0) 
			perror("gettimeofday()");
    	}

	pre_sieve(ssize, prnt, maxsvpr, &low, &nprimes, &totpr, fp);

	do_sieve(ssize, prnt, low, high, nprimes, &totpr, fp);

	if (fp != stdout) {
        	if (gettimeofday(&tpe, NULL) < 0) 
			perror("gettimeofday()");
		printf("Primes: " UL "\n", totpr);
        	printf("  Time: %.3f secs\n", (tpe.tv_sec - tps.tv_sec) \
                    + (float)(tpe.tv_usec - tps.tv_usec) / 1000000);
	}

	free(sieve);
	free(svprimes);
	return totpr;
}

/*
 * Pre sieving
 */
void
pre_sieve(Uns32 ssize, int prnt, Uns32 maxsvpr, Uns64 *plow, Uns32 *pnprimes, Uns64 *ptotpr, FILE *fp)
{
	Uns64	low;
	Uns32	s, totpr, nprimes;
	register Uns32	sc, i, sp;

	bzero(sieve, ssize>>4);
	s = sqrt(ssize);
	low = *plow;
	totpr = (Uns32)*ptotpr;
	nprimes = *pnprimes;

	for (i = 3; i <= s; i+=2) {
		if (!TEST(sieve, i)) {
			for (sc = i * i, sp = i<<1; sc < ssize; sc += sp ) 
				SET(sieve, sc);
		}
	}
	/* scan sieve for primes and print */
	if (low <= 2) {
		++totpr;
		if (prnt)
			fprintf(fp, "2\n");     
	}
	for (sc = 3; sc <= ssize; sc += 2) {
		if (!TEST(sieve, sc)) {
			if (sc <= maxsvpr) 
				svprimes[nprimes++] = sc;
			if (!TEST(sieve, sc) && sc >= low) {
					++totpr;
					if (prnt)
						print(sc, fp);
			}
		}
	}
	if (low <= ssize)
		low = ssize + 1;

	*plow = low;
	*ptotpr = (Uns64)totpr;
	*pnprimes = nprimes;
}

/*
 * Main sieving.
 * Strike out all the multiples of sieving primes in a page. 
 * The numbers remaining are primes.
 */
void
do_sieve(Uns32 ssize, int prnt, Uns64 low, Uns64 high, Uns32 nprimes, Uns64 *ptotpr, FILE *fp)
{
	Uns32		diff, i, s;
	Uns64		totpr, factor, pgend, inc, j, p;
	register Uns32	sp, sc;

	totpr = *ptotpr;
	p = (high - low) / ssize;
	inc = p / 100 + 1;

	for (j = 0; low < high; low = pgend + 1, j++) {

		if (high - low > ssize)
			pgend = low + ssize - 1;
		else
			pgend = high;

		bzero(sieve, ssize>>4);
		diff = pgend - low; 
		s = (Uns32)sqrt(pgend);

		for (i = 0; i < nprimes; ++i) {
			sp = svprimes[i];
			if (s < sp)
				break;
			factor = low / sp;
			if (low % sp) {
				factor++;
				factor += !(factor & 1);
			}
			for (sc = factor * sp - low, sp <<= 1; sc <= diff; sc += sp) {
				SET(sieve, sc);
			}
		}

		for (sc = 0; sc <= diff; sc += 2) {
			if (!TEST(sieve, sc)) {
					++totpr;
					if (prnt)
						print(sc + low, fp);
			}
		}

		if (pgend >= high)
			break;

		if(!prnt) {
			if (j % inc == 0) {
				printf("\r%2d%%", (int)((float)j / p * 100));
				fflush(stdout);
			}
		}
	}

	putchar('\r');
	*ptotpr = (Uns64)totpr;
}

/* 
 * print the primes 
 */
inline void
print(Uns32 num, FILE* fp) {						
	Uns32	digits[19];
	int 	len;

	len = 0;					
	do {						
		digits[len++] = num % 10;		
		num /= 10;				
   	} while (num);					
	for (--len; len >= 0; --len)			
		putc('0' + digits[len], fp);		
	putc('\n', fp);					
}
